////////////////////////////////////////////////////////////////////////////////

#include "console.h"
#include "product.h"
#include <ctype.h>
#include <string.h>
#include "hardware.h"
#include "errors.h"
#include "string_utils.h"
#include "syscmds.h"
#include "uart_cresnet_slave.hpp"
#include "cresnet_slave.h"
#include "memorymanager.h"
#include "ethernet.h"
#include "gateway.h"
#include <ctype.h>
#include <stdio.h>
#include "serialport.h"
#include "StreamMgr.h"

//Heap debugging
#pragma section = "HEAP"
#define HEAP_TEST_BLOCK      32

extern "C"
{
    extern UINT32 GetOsHeapSize(void);
    extern size_t vPortGetNextFreeByte( void );
}

const STRNUMPAIR GlobalHelpTypes[] =
{
    {"TEST", SYSTEM_TEST_HELP },
    {0, 0}                              // terminate the list
};

////////////////////////////////////////////////////////////////////////////////

#define BYTE_MEMORY_ACCESS  1
#define WORD_MEMORY_ACCESS  2
#define LONG_MEMORY_ACCESS  4
extern BOOL NVLChanged;
extern BOOL NVLSaved;

////////////////////////////////////////////////////////////////////////////////

INT32 InfoCmd(UINT32 ignore, char *cmd)
{
    //char PrefixString[16];
    //unsigned long   SysID;

    if ( cmd && *cmd == '?')                   // print help string
    {
        DmConsolePrintf("INFO\r");
        return 0;
    }

    DmConsolePrintf("  Category:          DM\r");
    DmConsolePrintf("  FirmwarePrefix:    %s\r", ProductGetFirmwareId());
#ifdef ETHERNET_SUPPORTED  // Only needed for Ethernet
    DmConsolePrintf("  Ethernet Ports:    1\r");
    DmConsolePrintf("  Max IP Entries:    1\r");
    DmConsolePrintf("  IP Masters:        1\r");
#endif
    ProductPrintInfo();

    return 0;
}

INT32 ReadWritePhysicalMemory(UINT32 WriteFlag, char *cmd)
{
#ifdef MEM_SUPPORTED
    char *p;
    unsigned long   MemoryAddress;
    unsigned char   *BytePointer;
    unsigned short  *WordPointer;
    unsigned long   *LongPointer;
    unsigned long   Data;
    int AccessType = LONG_MEMORY_ACCESS;
    char *endptr ;
    int i;
    int z;
    int q;
    unsigned char PrintData[16];

    if ( !(*cmd) || (!cmd) || (*cmd == '?') )     // no params, show cmd format
    {
        if ( WriteFlag == 0 )
        {
            DmConsolePrintf("MEMREAD [BYTE | WORD | LONG] address [length]\r");
            DmConsolePrintf("\t[BYTE | WORD | LONG] - memory access type (default to LONG)\r");
            DmConsolePrintf("\taddress    - the address for the memory access\r");
            DmConsolePrintf("\t[length]   - number of elements to show\r");
        }
        else
        {
            DmConsolePrintf("MEMWRITE [BYTE | WORD | LONG] address data\r");
            DmConsolePrintf("\t[BYTE | WORD | LONG] - memory access type (default to LONG)\r");
            DmConsolePrintf("\taddress    - the address for the memory access\r");
            DmConsolePrintf("\tdata       - hex numbers representing the data separated by spaces \r");
        }
        return 0;
    }

    // decode argument string
    p = strtok(cmd," ,") ;
    if ( !p )
    {
        DmConsolePrintf("Error parsing memory access or address\r");
        return -1;
    }
    if ( isalpha(*p) )
    {
        LocalConvertToUpper(p);
        if ( strncmp(p, "BYTE", 4) == 0 )
            AccessType = BYTE_MEMORY_ACCESS;
        else if ( strncmp(p, "WORD", 4) == 0 )
            AccessType = WORD_MEMORY_ACCESS;
        else if ( strncmp(p, "LONG", 4) == 0 )
            AccessType = LONG_MEMORY_ACCESS;

        p = strtok(NULL," ,");
        if ( !p )
        {
            DmConsolePrintf("Error parsing memory access\r");
            return -1;
        }
    }
    MemoryAddress = (unsigned long)strtoul(p, &endptr, 0);

    if ( WriteFlag == 0 )               // read memory
    {
        p = strtok(NULL," ,") ;
        if ( p )
            Data = (unsigned long)strtoul(p, &endptr, 0);
        else
            Data = 1;

        for ( i = 0; i < Data; i++)
        {
            if ( (i % (16/AccessType)) == 0)
            {
                DmConsolePrintf("\r\n%x :", (MemoryAddress + i*AccessType));
            }

            switch ( AccessType )
            {
                case BYTE_MEMORY_ACCESS:
                    BytePointer = (unsigned char *)MemoryAddress;
                    DmConsolePrintf(" %02x", *(BytePointer + i));
                    if(*(BytePointer+i) > 0x20 && *(BytePointer+i) < 0x7F)
                        PrintData[i%(16/BYTE_MEMORY_ACCESS)] = *(BytePointer + i);
                    else
                        PrintData[i%(16/BYTE_MEMORY_ACCESS)] = '.';
                    break;
                case WORD_MEMORY_ACCESS:
                    WordPointer = (unsigned short *)MemoryAddress ;
                    DmConsolePrintf(" %04x", *(WordPointer + i));
                    break;
                case LONG_MEMORY_ACCESS:
                    LongPointer = (unsigned long *)MemoryAddress;
                    DmConsolePrintf(" %08x", *(LongPointer + i));
                    break;
            }

            if(AccessType == BYTE_MEMORY_ACCESS)
            {
                BOOL doit=0;
                int count=0;
                if(i%16==15)
                {
                    count=16;
                    doit=1;
                }
                else if(i==Data-1)
                {
                    count=i%16+1;
                    for(z = 0 ; z < 16-count ; z++)
                    {
                        DmConsolePrintf("   ");
                    }
                    doit=1;
                }

                if(doit)
                {
                    DmConsolePrintf("     ");
                    for(q = 0 ; q < count ; q++)
                    {
                        DmConsolePrintf("%c", PrintData[q]);
                    }
                }
            }
        }
        DmConsolePrintf("\r");
    }
    else
    {
        p = strtok(NULL," ,") ;
        if ( !p )
        {
            DmConsolePrintf("Error parsing write data\r");
            return -1;
        }
        Data = (unsigned long)strtoul(p, &endptr, 0);
        switch ( AccessType )
        {
            case BYTE_MEMORY_ACCESS:
                BytePointer = (unsigned char *)MemoryAddress;
                *BytePointer = Data;
                break;
            case WORD_MEMORY_ACCESS:
                WordPointer = (unsigned short *)MemoryAddress;
                *WordPointer = Data;
                break;
            case LONG_MEMORY_ACCESS:
                LongPointer = (unsigned long *)MemoryAddress;
                *LongPointer = Data;
                break;
        }
    }
    return 0;
#else
    return 0;
#endif
}




/*
 * \author    Dariusz Rymsza
 * \brief     find string in memory using brute force method
 * \date      8/30/2010
 * \param     start address to search from
 * \param     end address to search to
 * \param     string to search for
 * \param     length of the string to find
 * \return    address of the found string or -1 if not found
 * \retval    address of the found string or -1 if not found
*/
static INT32 FindByteStringInMemory(UINT32 startAddr, UINT32 endAddr, char *searchString, UINT8 searchStringLen)
{
  bool bFound;
  UINT8 stringIdx;
  UINT32 i;

  bFound = false;

  stringIdx = 0;

  for (i = startAddr; i < endAddr; i++)
  {
    if (*(UINT8*)(i) == searchString[stringIdx++])
    {
      if (stringIdx == searchStringLen) // whole string found
      {
        bFound = true;
        break;
      }
    }
    else
      stringIdx = 0;  // restart matching from the beginning of the string
  }

  if (bFound)
    return (i - searchStringLen + 1);
  else
    return -1;
}
/**
 * \author    Dariusz Rymsza
 * \brief     returns the version number of bootloader
 * \date      8/30/2010
 * \param     pointer to text buffer
 * \param     length of buffer
 * \return    void
 * \retval    none
**/
// those have to be declared in the global area, otherwise they will not go into the flash
static char const searchStringBootloaderName[] = {'L','O','A','D','E','R',' ','[','v','0','0'};
static char const searchStringBootloaderVer[]  = {'.','0','0','0','0','.'};

#ifndef STM32F2XX
#pragma section="FirmwareId"
//extern void *__ICFEDIT_region_ROM_start__;
//static void * const pEndOfBootloaderFlash = &__ICFEDIT_region_ROM_start__;
static void * const pEndOfBootloaderFlash = __section_begin("FirmwareId"); // bootloader area has to end before application start
#else
static char const searchStringPreBootName[] = {'P','R','E','-','B','O','O','T',' ','[','v','0','0'};
#endif

void GetBootloaderVersion(char * p, UINT32 len, char *cmd, UINT8 *pStartAddress)
{
  // brute force search for booloader version using identifying version string mark '0000.'
  INT32 stringIdx, stringIdxOrg, stringIdxName;
  UINT8 searchStringLen = 0;
  
// Devices with stm32f2core may have a preboot section
#ifdef STM32F2XX
  UINT32 pEndOfBootloaderFlash = 0;
  
  // Search for preboot or the bootloader version based on the address passed in
  if ((UINT32)pStartAddress == g_FlashMap[g_FlashConfig].BtlStartAddr)
  {
    pEndOfBootloaderFlash = ((UINT32)pStartAddress) + g_FlashMap[g_FlashConfig].BtlSize;
    searchStringLen = sizeof(searchStringBootloaderName);
    stringIdx = stringIdxOrg = FindByteStringInMemory((UINT32)pStartAddress, pEndOfBootloaderFlash, (char*)searchStringBootloaderName, searchStringLen);
  }
  else
  {
    pEndOfBootloaderFlash = ((UINT32)pStartAddress) + g_FlashMap[g_FlashConfig].PreBootSize;
    searchStringLen = sizeof(searchStringPreBootName);
    stringIdx = stringIdxOrg = FindByteStringInMemory((UINT32)pStartAddress, pEndOfBootloaderFlash, (char*)searchStringPreBootName, searchStringLen);
  }
  
#else
  searchStringLen = sizeof(searchStringBootloaderName);
  stringIdx = stringIdxOrg = FindByteStringInMemory((UINT32)pStartAddress, (UINT32)pEndOfBootloaderFlash, (char*)searchStringBootloaderName, searchStringLen);
#endif
  
  if (stringIdx > 0)
  {
    // back up to the beginning of the bootloader name
    while (stringIdx > 0x0 && *(UINT8*)(stringIdx) >= ' ' && *(UINT8*)(stringIdx) < 0x80)
      stringIdx--;

    // most likely we are before the name now, probably terminating null from another string or some other area
    // advance forward to the first character of the bootloader name
    stringIdx++;

    // now skip over leading spaces and dashes
    while (stringIdx < stringIdxOrg && (*(UINT8*)(stringIdx) == ' ' || *(UINT8*)(stringIdx) == '-'))
      stringIdx++;

    stringIdxName = stringIdx;  // remember name position

    // now find the resto of the version string
    searchStringLen = sizeof(searchStringBootloaderVer);
    stringIdx = FindByteStringInMemory((UINT32)pStartAddress, (UINT32)pEndOfBootloaderFlash, (char*)searchStringBootloaderVer, searchStringLen);

#ifndef STM32F2XX
    if (stringIdx > 0)
    {
     // snprintf(p, len, "%s1%s\r\n ", (UINT8*)stringIdxName, (UINT8*)stringIdx);
      snprintf(p, len, "%s1", (UINT8*)stringIdxName);
      len -= (strlen(p) + 1); // +1 for terminating null
      // since old bootloaders were creating version string differently, we need to extract it from the sprintf format string
      while (len--)
      {
        strncat(p, (char const*)stringIdx, 1);
        if (*(UINT8*)stringIdx++ == ']')        // copy up to the last bracket ']'
          break;
      }
    }
#else

    // Set ID to begining of version number
    while ( (stringIdx > 0) && ( (*(UINT8*)(stringIdx) < '0') || (*(UINT8*)(stringIdx) > '9') ) )
      stringIdx--;

    // Make sure version name and numbering both exist
    if ( (stringIdxName > 0) && (stringIdx > 0) )
    {
      snprintf(p, len, "%s", (char*)stringIdxName);

      char *pEndOfString = NULL;

      // Search for ',' which signifies the end of the version numbering string portion to copy
      pEndOfString = (char*) memchr((char*)stringIdx, ',', len);

      if (pEndOfString != NULL)
      {
        // Append string up to ','
        strncat(p, (char*)stringIdx, (pEndOfString - (char*)stringIdx));
        strncat(p, "]", 1); // Append closing bracket
      }
    } // if ( (stringIdxName > 0) && (stringIdx > 0) )
#endif
  }
}

INT32 VersionCmd(UINT32 ignore, char *cmd)
{
    char c, temp[81];

    FirmwareGetName(temp, sizeof(temp), 0);

    DmConsolePrintf("%s\r", temp);

    if( *cmd )
    {
        c = toupper(cmd[1]);

        if (c == 'V' || c == 'O')
        {
            // if verbose flag is set report versions of any additional files to upload
            // e.g. FPGA, DSP
            *temp = 0;
            ProductGetVersion(temp, sizeof(temp), cmd);
            if (*temp != NULL)
                DmConsolePrintf("%s\r", temp);
            // print bootloader version

            UINT8 *pBLBaseAddr;
#ifdef STM32F2XX            
            pBLBaseAddr = GetBLBase();
#else
            pBLBaseAddr = 0;
#endif
            *temp = 0;
            GetBootloaderVersion(temp, sizeof(temp), cmd, pBLBaseAddr);
            if (*temp != NULL)
                DmConsolePrintf("%s\r", temp);
#ifdef STM32F2XX
            // Reuse the ptr for preboot to get the base address, if it's not invalid then there's
            // a preboot in on this device
            pBLBaseAddr = GetPrebootBase(); 
            if (pBLBaseAddr != (UINT8*)INVALID_FLASH_ADDR)
            {
                *temp = 0;
                GetBootloaderVersion(temp, sizeof(temp), cmd, pBLBaseAddr);
                if (*temp != NULL)
                    DmConsolePrintf("%s\r", temp);
            }
#endif
        }
    }

    return 0;
}

//Bugzilla 53181 fix
//PrepareReboot function pointer
PF_DM_PREPARE_REBOOT pfDmPrepareReboot = 0;

void (*pFunction_SendRebootCmd)(void)= 0;
INT32 RebootCmd(UINT32 ignore, char *cmd)
{
	int timeout = 0;

    if ( cmd && *cmd == '?')                   // print help string
    {
        DmConsolePrintf("REBOOT\r");
        DmConsolePrintf("\tNo parameters\r");
        return 0;
    }
#ifdef STM32F2XX
    else if (stricmp(cmd,"WARM") == 0)
    {    
        AppBLoaderSharedData * pAppBLoaderSharedData = (AppBLoaderSharedData*)(GetBootCmdPtr() - sizeof(AppBLoaderSharedData) - RAM_EXCEPTION_TABLE_LENGTH);
        
        pAppBLoaderSharedData->warmReboot[0] = 'W';
        pAppBLoaderSharedData->warmReboot[1] = 'A';
        pAppBLoaderSharedData->warmReboot[2] = 'R';
        pAppBLoaderSharedData->warmReboot[3] = 'M';
    }
#endif

    if( pfDmPrepareReboot )
    {
        pfDmPrepareReboot();
    }

    DmConsolePrintf("Rebooting...\r");

    //If application needs to send the reboot command to the downstrean devices,
    //then do it here.
    if(pFunction_SendRebootCmd)
    {
      pFunction_SendRebootCmd();
      HwDelayMsec(1000);
    }

#ifdef ETHERNET_SUPPORTED
    closeAllEthConsole(false);
    CloseAllGatewayConnections();
    SetDhcpTimeStamp();
    HwDelayMsec(200); // give it a chance to transmit

    if (pEtherParam->UseDHCP == FALSE)
      ReleaseDhcp(); // Function has address checking inside
#endif

    for(timeout=0; timeout<300; timeout++ )
    {
       if (!NVLChanged && NVLSaved)
         break;

       // delay to finish console feedback
       // also wait until NVL is updated
        HwDelayMsec(10);
    }
#ifdef APPLICATION_CLEANUP
	 //	Note: to enable the application cleanup, set the APPLICATION_CLEANUP parameter in the Crestore IAR project file and define this function in the
	//	application. It was done this way not as a call or a pointer due to RAM and ROM space limitations in some DM cards.
	//	This function would prepare application cleanup such as memory release, stop processing or save eeprom parameters etc...
	ApplicationCleanup();
#endif
    DmRebootNotFatal();

    // will never get here...
    return 0;
}

// was SetCharacterEcho
INT32 EchoCmd(UINT32 ignore, char *cmd)
{
    if (*cmd == '?')                   // print help strilng
    {
        DmConsolePrintf("ECHO [ON | OFF]\r");
        DmConsolePrintf("\tNo parameter - displays current setting\r");
        return 0;
    }

    LocalConvertToUpper(cmd);           // convert parameter to uppercase
    strtok(cmd," \r\n\t");    // this will put a NULL after the parameter

    if (*cmd == 0)
    {
        DmConsolePrintf("Character Echo is %s\r",(DmConsoleGetEcho()) ? "ON" : "OFF");
        return 0;
    }

    if ( strcmp(cmd,"ON") == 0 )
    {
        DmConsoleSetEcho(1);
    }
    else if ( strcmp(cmd,"OFF") == 0 )
    {
        DmConsoleSetEcho(0);
    }

    return 0;
}

INT32 RestoreCmd(UINT32 ignore, char *cmd)
{
  int timeout=0;
    char* pText;
    char ResponseString[32] = {0x00};

    if ( cmd && *cmd == '?')                   // print help strilng
    {
        DmConsolePrintf("RESTORE Y\r");
        return 0;
    }
    DmConsolePrintf("WARNING: This command will set factory defaults and then reboot.\r");

    //Since this function is not implemented, simply require an extra parameter
    //DmConsoleGetUserResponse("Do you want to continue (Y or N)? ",ResponseString, sizeof(ResponseString));

    //Check the first parameter for yes/no
    pText = strtok(cmd, " ");
    if ( pText )
    {
        //Copy the argument to the response string
        ResponseString[0] = *pText;
    }

    LocalConvertEntireStringToUpper(ResponseString);
    if ( ResponseString[0] != 'Y')
    {
        DmConsolePrintf("Type RESTORE Y to confirm\r");
        return -1;
    }

    RestoreFactoryDefaults();
    DmConsolePrintf("Factory Defaults have been restored.  Rebooting....\r");

    for(timeout=0; timeout<300; timeout++ )
    {
       if (!NVLChanged && NVLSaved)
         break;

       // delay to finish console feedback
       // also wait until NVL is updated
        HwDelayMsec(10);
    }

#if 0
    if((timeout >= 300) && ( NVLChanged || !NVLSaved))
    {
    	asm("BKPT 0");
    }
#endif
    DmRebootNotFatal();

    // never get here

    return 0;
}

// Located in product specific project for 3 series
#ifndef SERIES_3_CONTROL
INT32 SelfTestCmd(UINT32 ignore, char *cmd)
{
    BOOL local = 1;

    if ( cmd && *cmd == '?')                   // print help strilng
    {
        DmConsolePrintf("SELFTEST\r");
//        DmConsolePrintf("\tNo parameter\r");
        return 0;
    }
    local = DmConsoleIsLocal();
    SelfTest(local);

    return 0;
}
#endif

INT32 UptimeCmd(UINT32 ignore, char *cmd)
{
    UINT32 upTimeMsec = HwGetMsec();
    UINT16 days;
    UINT8 hours, minutes, sec;

    if ( *cmd == '?' )     // show cmd format
    {
        DmConsolePrintf("UPTIME\r");
//        DmConsolePrintf("\tno parameters needed\r");
        return 0;
    }

    DmConsolePrintf("The system has been running for %u msec\r", upTimeMsec);

    days = upTimeMsec/(60*60*1000*24);
    upTimeMsec %= (60*60*1000*24);
    hours = upTimeMsec/(60*60*1000);
    upTimeMsec %= (60*60*1000);
    minutes = upTimeMsec/(60*1000);
    upTimeMsec %= (60*1000);
    sec = upTimeMsec/(1000);
    upTimeMsec %= (1000);
    DmConsolePrintf(" or %u days %02d:%02d:%02d.%03d\r",
    days, // days
    hours, // hours
    minutes, // min
    sec, // sec
    upTimeMsec) ; // msec

    return 0;
}
/**
 * \author    Larry Salant
 * \brief     if number is specified, save it to nonvolatile memory and
 *            update the PnP address for cresnet;  otherwise return current value
 * \date      3/7/2008
 * \param     pointer to command line arguments
 * \return    void
 * \retval    none
**/
INT32 TSIDCmd(UINT32 ignore, char * cmd)
{
	//	fix bugzilla# 39447
	typedef	struct	{
		TYPE_ARG_STRUCT arg[1];
	}	TYPEARG;
        TYPEARG dstr;
	TYPEARG *parg = &dstr;
	UINT8   argc;
	INT32	res = 0;

	GetCmdArguments ((INT8*)cmd, parg->arg, &argc, 1);
	if (IsCmdHelp(parg->arg[0].argv))	{
		//	display TSID help
		DmConsolePrintf("TSID [touch_settable_id]\r");
        DmConsolePrintf("\ttouch_settable_id - ID of the device (in hex).\r");
	}	else if (argc == 0)	{
		//	display TSID
		DmConsolePrintf("Touch Settable ID = %08X\r", UartCresnetSlaveGetPPNAddr());
	}	else if ((parg->arg[0].type != CONSOLE_PAR_TYPE_NUM) || (parg->arg[0].parh == CNET_PLUG_PLAY_BROADCAST_PPN))	{
        //	display error
		DmConsolePrintf("ERROR: Invalid Touch Settable ID\r");
        res = -1;
	}	else	{
		//  set the ppn for cresnet & Save TSID in non-volatile memory
		SetPNPNumber(parg->arg[0].parh);
	}

    return res;
}

// jump into bootloader
INT32 SystemCmd(UINT32 ignore, char *ncmd)
{
    CresnetBootloaderRequest(1, 0);
    return 0;
}

INT32 HelpCmd(UINT32 ignore, char *cmd)
{
    UINT32 helpType;
    char temp[81];


    if ( *cmd == '?')                   // print help string
    {

        if(NumberGetString((STRNUMPAIR *)HelpTypes, ETHERNET_HELP, temp, sizeof(temp)) == 0)
        {
            DmConsolePrintf("HELP [ALL|DEV|ETHER|SYS]\r");
        }

        else
        {
            DmConsolePrintf("HELP [ALL|DEV|SYS]\r");
        }
        DmConsolePrintf("\tNo parameter - detail information on help system\r");
        return 0;
    }
    if ( *cmd == '\0' )                 // no parameter
    {
        DmConsolePrintf("HELP - Detailed help available.  Type a command followed by\r");
        DmConsolePrintf("       a space and '?' to see options for that command\r");
        DmConsolePrintf("       (i.e. HELP ?)\r");
        DmConsolePrintf("HELP ALL will display a list of all commands.\r");
        DmConsolePrintf("HELP DEVICE will display a list of cmds specific to this device.\r");
        if(NumberGetString((STRNUMPAIR *)HelpTypes, ETHERNET_HELP, temp, sizeof(temp)) == 0)
          DmConsolePrintf("HELP ETHERNET will display a list of Ethernet commands.\r");
#ifdef FILE_SYSTEM
        if(NumberGetString((STRNUMPAIR *)HelpTypes, FILE_HELP, temp, sizeof(temp)) == 0)
        {
            DmConsolePrintf("HELP FILE will display a list of commands for file operations.\r");
        }
#endif
        DmConsolePrintf("HELP SYSTEM will display a list of general system commands.\r");
#ifdef AUDIO_SYSTEM
        if(NumberGetString((STRNUMPAIR *)HelpTypes, AUDIO_HELP, temp, sizeof(temp)) == 0)
        {
            DmConsolePrintf("HELP AUDIO will display a list of audio/video commands.\r");
        }
#endif
        return 0;
    }
    // convert cmd string to upper case
    LocalConvertToUpper(cmd);

    //first check global help types
    if( StringGetNumber((STRNUMPAIR *)GlobalHelpTypes, cmd, &helpType, 0) != 0 )
    {
        //then card-specific help types
        if(StringGetNumber((STRNUMPAIR *)HelpTypes, cmd, &helpType, 0) != 0)
        {
            DmConsolePrintf("Unknown Parameter: %s\r", cmd);
            return -1;
        }
    }

    ShowHelpMenu(helpType);

    return 0;
}

extern "C" {
  extern void vTaskList( signed char *pcWriteBuffer, unsigned long max );
}

#define DEFAULT_TRACE_BUFFER_SIZE 250
INT32 TasksShowCmd(UINT32 argc, char * cmd)
{
#ifndef STM32F2XX  
    char *tBuff;
    char *p, *endptr;
    UINT32 sizeWanted = DEFAULT_TRACE_BUFFER_SIZE;
    
    if ( cmd && (*cmd != 0))                   // print help strilng
    {
        // decode argument string
       p = strtok(cmd," -.,;:") ;
       if ( !p )
           return -1;

       sizeWanted = strtoul(p, &endptr, 0);
    }

    if (MemMgr && (tBuff = (char *)MemMgr->GetBlock(sizeWanted)) != 0)
    {
      vTaskList( (signed char *)tBuff, sizeWanted)  ;

      tBuff[sizeWanted-1] = 0;  // make sure it's NULL terminated

      DmConsolePuts(tBuff);

      MemMgr->FreeBlock((UINT8*)tBuff) ;
    }
    else
    {
        DmConsolePrintf("Cannot allocate \r");
    }
#else
    // F2 devices has specific function for displaying tasks info
    OsPrintTaskInfoAdvanced();
#endif
    
    return 0;
}

/**
 * \author      Adolfo Velasco
 *
 * \brief       Performs the heap test
 *
 * \date        09/05/08
 *
 * \param       none
 *
 * \return      none
 *
 * \retval      none
 *
 */
#define HEAP_REPORT_BLOCK_COUNT 4

/*
  DR 2/10/11 - this is a nasty hack to get some more info out of freeRTOS
  I didn't want to change rtos source to make this structure exposed so it's defined here again
*/
#ifdef STR91X_IAR

typedef struct A_BLOCK_LINK
{
	struct A_BLOCK_LINK *pxNextFreeBlock;	/*<< The next free block in the list. */
	size_t xBlockSize;						/*<< The size of the free block. */
} xBlockLink;

extern xBlockLink xStart, xEnd;

#endif // STR91X_IAR

INT32 HeapTestCmd(UINT32 argc, char * cmd)
{
    UINT32 lSize, lSizeTotal, lSizeDec;
    UINT8* pEndHeap;

    //Allocate the end heap
    pEndHeap = (UINT8*) malloc(HEAP_TEST_BLOCK);

    //Verify allocation
    if ( !pEndHeap )
    {
        DmConsolePrintf( "Unable to allocate %d bytes\r", HEAP_TEST_BLOCK );
    }
    else
    {
      //Device heap
      DmConsolePrintf("Device heap\r");

      //Print the current used heap
      lSize = pEndHeap - ( (UINT8*)__section_begin("HEAP") );
      DmConsolePrintf("Used: 0x%x (%d)\r", lSize, lSize );

      //Free the heap test block
      free(pEndHeap);

      //Total
      lSizeTotal = ( (UINT8*)__section_end("HEAP") ) - ( (UINT8*)__section_begin("HEAP") );
      DmConsolePrintf("Total: 0x%x (%d)\r", lSizeTotal, lSizeTotal );

      //get total contiguous unused
      lSize    = lSizeTotal - lSize;
      // we don't know how badly fragmented is the heap, the object of the game is to find size of the largest contiguous block
      // try to allocate as big chunk as possible, this way we avoid allocation of one of the smaller holes
      // and treating it as the largest contiguous block
      lSizeDec = lSizeTotal;//lSize;
      pEndHeap = NULL;

      // find four largest heap blocks
      UINT32 heapDecrement;
      heapDecrement = 4;

      UINT8* pHeapStart = (UINT8*)__section_begin("HEAP");
      UINT8* pHeapPtr[HEAP_REPORT_BLOCK_COUNT] = {0,0,0,0}; // pointers to allocated blocks
      int i;

      for (i = 0; i < HEAP_REPORT_BLOCK_COUNT; i++)
      {
        lSizeDec = lSizeTotal;  // try to grab biggest block

        while( !(pHeapPtr[i] = (UINT8*) malloc(lSizeDec) ) && (lSizeDec > heapDecrement))
        {
           lSizeDec -= heapDecrement; //decrease block size we are looking for
        }
        if ((i == 0) || (pHeapPtr[i]))  // always print first block status, consecutive only if allocated
        {
          if (!(pHeapPtr[i]))
          {
            lSizeDec = 0;
            pHeapPtr[i] = pHeapStart;
          }

          DmConsolePrintf("Available: 0x%x (%d) @ 0x%x (%d)\r", lSizeDec, lSizeDec, pHeapPtr[i] - pHeapStart, pHeapPtr[i] - pHeapStart);
        }
      }

      for (i = 0; i < HEAP_REPORT_BLOCK_COUNT; i++)
      {
        if (pHeapPtr[i])
          free(pHeapPtr[i]);
      }

    }

    //OS heap
    DmConsolePrintf("\r\nOS heap\r");

    //Total
    lSizeTotal = GetOsHeapSize();

    //Used
    lSize = (UINT32) vPortGetNextFreeByte();
    DmConsolePrintf("UNused: 0x%0x (%d)\r",
#ifdef IAR_ARM_CM3
                    lSizeTotal - lSize, lSizeTotal - lSize);
#else
                    lSize, lSize);
#endif // IAR_ARM_CM3

    //Total
    DmConsolePrintf("Total: 0x%0x (%d)\r", lSizeTotal, lSizeTotal);

#ifdef STR91X_IAR
    // walk OS heap, printing block sizes
    xBlockLink  *pStart;
    pStart = &xStart;

    while ((pStart = pStart->pxNextFreeBlock) != &xEnd)
    {
      DmConsolePrintf("Available: 0x%x (%d) @ 0x%x\r", pStart->xBlockSize, pStart->xBlockSize, pStart);
    }
#endif // STR91X_IAR

    // if memory manager available
    if (MemMgr)
        // Print the number of buffers never used
        MemMgr->PrintUnused();

    return 0;
}

/**
 * \author      Adolfo Velasco
 * \brief       Performs the join test
 * \date        09/05/08
 * \param       none
 * \return      none
 * \retval      none
 *
 */
INT32 JoinTestCmd(UINT32 argc, char * cmd)
{
    UINT8 i;
    CresnetJoinState* pJoinState;

    for ( i = 0; i < GetNumStreams(); i++ )
    {
        //Get the join state
        pJoinState = GetState(i);
        if ( pJoinState )
        {
            DmConsolePrintf("Stream%d - Digital usage (%d of %d)\r", i+1, pJoinState->GetUsedDigitalCount(), pJoinState->GetMaxDigitalCount());
            DmConsolePrintf("Stream%d - Analog usage (%d of %d)\r", i+1, pJoinState->GetUsedAnalogCount(), pJoinState->GetMaxAnalogCount());
            DmConsolePrintf("Stream%d - Serial usage (%d of %d)\r", i+1, pJoinState->GetUsedSerialCount(), pJoinState->GetMaxSerialCount());
        }
    }

    return 0;
}

/**
 * \author      Adolfo Velasco
 * \brief       Performs the buffer test
 * \date        06/18/09
 * \param       none
 * \return      none
 * \retval      none
 *
 */
INT32 BufferTestCmd(UINT32 argc, char * cmd)
{
    CDMUartCresnetSlave* pUartSlave;

    //Get the CresnetSlaveUart
    pUartSlave = GetCresnetSlaveUart();
    if ( pUartSlave )
    {
        DmConsolePrintf("CnetSlaveUart\r");
        if ( pUartSlave->m_pStream )
        {
            DmConsolePrintf("OutgoingCmdQueue - FullCount:%d\r", pUartSlave->m_pStream->pOutGoingCmdQueue->GetFullCount());
            DmConsolePrintf("OutgoingCmdQueue - MaxDepth:%d\r", pUartSlave->m_pStream->pOutGoingCmdQueue->GetMaxDepth());
        }
        else
        {
            DmConsolePrintf("OutgoingCmdQueue - NULL\r");
        }
        if ( pUartSlave->m_MidData )
        {
            DmConsolePrintf("PendingToController - FullCount:%d\r", pUartSlave->m_MidData->m_pPendingToController->GetFullCount());
            DmConsolePrintf("PendingToController - MaxDepth:%d\r", pUartSlave->m_MidData->m_pPendingToController->GetMaxDepth());
        }
        else
        {
            DmConsolePrintf("PendingToController - NULL\r");
        }
    }
    else
    {
        DmConsolePrintf("Unable to access CnetSlaveUart\r");
    }

    //Get the midpoint UART
    pUartSlave = GetDMMidpointUart(0);
    if ( pUartSlave )
    {
        DmConsolePrintf("MidPointSlaveUart\r");
        if ( pUartSlave->m_pStream )
        {
            DmConsolePrintf("OutgoingCmdQueue - FullCount:%d\r", pUartSlave->m_pStream->pOutGoingCmdQueue->GetFullCount());
            DmConsolePrintf("OutgoingCmdQueue - MaxDepth:%d\r", pUartSlave->m_pStream->pOutGoingCmdQueue->GetMaxDepth());
        }
        else
        {
            DmConsolePrintf("OutgoingCmdQueue - NULL\r");
        }
        if ( pUartSlave->m_MidData )
        {
            DmConsolePrintf("PendingToController - FullCount:%d\r", pUartSlave->m_MidData->m_pPendingToController->GetFullCount());
            DmConsolePrintf("PendingToController - MaxDepth:%d\r", pUartSlave->m_MidData->m_pPendingToController->GetMaxDepth());
        }
        else
        {
            DmConsolePrintf("PendingToController - NULL\r");
        }
    }
    else
    {
        DmConsolePrintf("Unable to access MidPointSlaveUart\r");
    }

    return 0;
}

// external memory test
INT32 MemoryTest(UINT32 ignore, char *cmd)
{
    char *p;
    void *MemoryAddressPtr;
    unsigned long   Data;
    char *endptr ;
    unsigned char   *BytePointer;
    int i;
    bool bAlloc = false;


    if ( !(*cmd) || (!cmd) || (*cmd == '?') )     // no params, show cmd format
    {
        DmConsolePrintf("MEMTEST [INT | EXT0 | EXT1 | EXT2 | EXT3] [length]\r");
        DmConsolePrintf("\t[INT | EXT]    - memory region (internal/external)\r");
        DmConsolePrintf("\t[length]   - number of elements to test\r");
        return 0;
    }

    // decode argument string
    p = strtok(cmd," ,") ;
    if ( !p )
    {
        DmConsolePrintf("Invalid arguments\r");
        return -1;
    }

    if ( isalpha(*p) )
    {
        LocalConvertToUpper(p);
        if ( strncmp(p, "INT", 3) == 0 )
          bAlloc = true;
        else if ( strncmp(p, "EXT0", 4) == 0 )
            MemoryAddressPtr = (void*)0x3C000000;
        else if ( strncmp(p, "EXT1", 4) == 0 )
            MemoryAddressPtr = (void*)0x38000000;
        else if ( strncmp(p, "EXT2", 4) == 0 )
            MemoryAddressPtr = (void*)0x34000000;
        else if ( strncmp(p, "EXT3", 4) == 0 )
            MemoryAddressPtr = (void*)0x30000000;
        else
        {
            DmConsolePrintf("Invalid arguments\r");
            return -1;
        }

        p = strtok(NULL," ,");
        if ( !p )
        {
            DmConsolePrintf("Invalid arguments\r");
            return -1;
        }
    }

    if ( p )
        Data = (unsigned long)strtoul(p, &endptr, 0);
    else
    {
        DmConsolePrintf("missing length\r");
        return -1;
    }

    if (bAlloc)
      MemoryAddressPtr = malloc(Data);

    if (!MemoryAddressPtr)
    {
      DmConsolePrintf("Cannot allocate %d bytes\r", Data);
      return (-1);
    }

    DmConsolePrintf("Testing from 0x%04x, %d bytes\r", (unsigned char*)MemoryAddressPtr, Data);

    BytePointer = (unsigned char *)MemoryAddressPtr;

    UINT8 data;

    // write zeroes
    for (i = 0; i < Data; i++)
    {
      *(BytePointer + i) = 0;
    }

    // read zero, write 1
    for (i = 0; i < Data; i++)
    {
      data = *(BytePointer + i);
      if (data != 0)
        DmConsolePrintf("%X != 0 (%X)\r", BytePointer + i, data);
      *(BytePointer + i) = 0xFF;
    }

    // read 1, write zero
    for (i = Data - 1; i >=0; i--)
    {
      data = *(BytePointer + i);
      if (data != 0xFF)
        DmConsolePrintf("%X != 0xFF (%X)\r", BytePointer + i, data);
      *(BytePointer + i) = 0x0;
    }

    if (MemoryAddressPtr <= (void*)0x020000)
      free(MemoryAddressPtr);

    return 0;
}
void (*pfuncDmHDBaseTUartConfig)(int bStream, UINT8 bUartConfigType, int cmdSet)= 0;
UINT8   g_bNumberOfDMHDBaseTUarts = 0;
/**
 * \author      Andrew Salmon
 * \brief       Force HDBaseT Link config
 * \date        10/07/13
 * \param       none
 * \return      none
 * \retval      none
 *
 */
INT32 DmHDBaseTUartConfigCmd(UINT32 ignore, char *cmd)
{
    char *endptr;
    UINT32 argc;
    char * argv[10];
    int cmdSet = 1;
    int bStream = STREAM_1;
    UINT8 bUartConfigType = DM_HDBASET_UART_CONFIG_TYPE_AUTO_DETECT;

    if( !pfuncDmHDBaseTUartConfig )
    {
        DmConsolePrintf(" HDBASETConfig Not Supported\r");
        return 0;
    }

    LocalConvertEntireStringToUpper(cmd);

    StringParseArgs(cmd, argv, sizeof(argv)/sizeof(char*), &argc);

    if(*cmd == '?' )
    {
hdbthelp:
        if( g_bNumberOfDMHDBaseTUarts > 1 )
        {
            DmConsolePrintf(" HDBASETConfig STREAM stream# [SERIAL|DMNET|AUTO]\r");
            DmConsolePrintf(" \tstream# is 0-based\r");
        }
        else
        {
            DmConsolePrintf(" HDBASETConfig [SERIAL|DMNET|AUTO]\r");
        }
        return 0;
    } 
    else
    {
        if( argc )
        {
            //Check if 1st input argument is "STREAM"
            if( !strcmp(argv[0],"STREAM") )
            {
                bStream = (BYTE)strtol(argv[1], &endptr, 0);
                if( !strcmp(argv[2],"SERIAL") )
                {
                    bUartConfigType = DM_HDBASET_UART_CONFIG_TYPE_SERIAL_PORT;
                }
                else if( !strcmp(argv[2],"DMNET") )
                {
                    bUartConfigType = DM_HDBASET_UART_CONFIG_TYPE_DMNET;
                }
                else if( !strcmp(argv[2],"AUTO") )
                {
                    bUartConfigType = DM_HDBASET_UART_CONFIG_TYPE_AUTO_DETECT;
                }
                else
                {
                    goto hdbthelp;
                }
            }
            else
            {
                if( !strcmp(argv[0],"SERIAL") )
                {
                    bUartConfigType = DM_HDBASET_UART_CONFIG_TYPE_SERIAL_PORT;
                }
                else if( !strcmp(argv[0],"DMNET") )
                {
                    bUartConfigType = DM_HDBASET_UART_CONFIG_TYPE_DMNET;
                }
                else if( !strcmp(argv[0],"AUTO") )
                {
                    bUartConfigType = DM_HDBASET_UART_CONFIG_TYPE_AUTO_DETECT;
                }
                else
                {
                    goto hdbthelp;
                }
            }
        }
        else
        {
            cmdSet = 0;
        }

        pfuncDmHDBaseTUartConfig(bStream, bUartConfigType, cmdSet);
    }
    return 0;
}
